﻿#include "precompiled.h"
#include "common.h"
#include "Camera.h"

#include "CameraProperties.h"
#include "Lens.h"
#include "Film.h"
#include "Aperture.h"
#include "Shutter.h"
#include "Texture.h"

#include "FStops.h"

#include "Entity.h"
#include "Transform.h"

using namespace DirectX;

namespace RTCam {

const float kLensMovementStep = 0.1f * kMmToMeters; // Distance by which to move a lens when lens movement functions are called

// Thresholds past which the focal distance steps will use larger values
const float kFocalDistanceThresholds[] = {
	0.2f,
	1,
	5,
	10,
	20,
	50,
	100,
	200
};
// Meters by which to change the focal distance of the camera
const float kFocalDistanceSteps[] = {
	0.02f,
	0.1f,
	0.25f,
	1,
	2,
	5,
	10,
	50
};
const float kLargestFocalDistanceStep = 50;				// The focal distance step size to use when none of the thresholds match

const float kShutterSpeedMultiplier = 2;				// Factor by which shutter speeds differ
const int kFilmSpeedMultiplier = 2;						// Factor by which film speeds differ
const float kLensFocalLengthStep = 10 * kMmToMeters;	// Difference in focal length of lenses being switched out
const float kCameraAnimationDuration = 0.2f;			// How long it takes camera settings to animate to their new values

Camera::Camera(const shared_ptr<Entity>& entity) :
	Component(entity),
	m_cameraCBufferData(),
	m_currentLensName("Default Lens"),
	m_currentFilmName("Default Film"),
	m_film(new Film()),
	m_lens(new Lens()),
	m_aperture(new Aperture()),
	m_shutter(new Shutter()),
	m_fNumberAnim(),
	m_lensDistanceAnim(),
	m_focalLengthAnim()
{
	// Initialize animations
	m_lensDistanceAnim.FinalVal = m_lens->GetLensDistance();
	m_fNumberAnim.FinalVal = m_aperture->GetFNumber();
	m_focalLengthAnim.FinalVal = m_lens->GetFocalLength();
}

Camera::~Camera(void)
{
}

float Camera::CalculateExposureModifier()
{
	// This is currently tuned according to the "Sunny 16 rule".
	// f/16 aperture, ISO 100, and 1/100 sec exposure should give a good exposure
	// with the equivalent of sunlight illuminating the scene
	const float normFNumber = 16.0f;
	const float normFilmSpeed = 100.0f;
	const float normShutterSpeed = 100.0f; // reciprocal of 1 / 100 is 100
	
	// Modifier from aperture area
	// (Twice the area = double the exposure)
	// Aperture area = PI * (focal length / 2 * f-Number) ^ 2
	float focalLength = m_lens->GetLensDistance();

	float apertureRadius = focalLength / (2 * m_aperture->GetFNumber());
	float apertureArea = kPi * apertureRadius * apertureRadius;

	float normRadius = focalLength / (2 * normFNumber);
	float normArea = kPi * normRadius * normRadius;
	
	float apertureModifier = apertureArea / normArea;

	// Modifier from film speed
	// (200 is twice the exposure of 100; 400 is twice as much as 200 and 4x as much as 100)
	int filmSpeed = m_film->GetFilmSpeed();
	float filmSpeedModifier = filmSpeed / normFilmSpeed;
	
	// Modifier from shutter speed
	float shutterSpeed = m_shutter->GetShutterSpeed();
	float shutterSpeedModifier = shutterSpeed * normShutterSpeed;

	float normalizedExposure = apertureModifier * filmSpeedModifier * shutterSpeedModifier;

	return normalizedExposure;
}

float Camera::CalculateDiagonalAngleOfView()
{
	return CalculateAngleOfView(kSensorSizeDiagonal);
}

float Camera::CalculateVerticalAngleOfView(float aspectHByW)
{
	auto filmSize = CalculateFilmSize(aspectHByW);
	return CalculateAngleOfView(filmSize.y);
}

XMFLOAT2 Camera::CalculateFilmSize(float aspectHByW)
{
	// y ^ 2 = diag ^ 2 - x ^ 2
	// aspectRatio is x / y
	// y ^ 2 = diag ^ 2 / ( 1 + aspectRatio ^ 2)
	float d2 = kSensorSizeDiagonal * kSensorSizeDiagonal;
	float OnePlusR2 = 1 + aspectHByW * aspectHByW;
	float y2 = d2 / OnePlusR2;
	float y = sqrtf(y2);

	// x ^ 2 = diag ^ 2 - y ^ 2
	float x = sqrtf(d2 - y2);

	return XMFLOAT2(x, y);
}


void Camera::MoveLensForward()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initial = m_lens->GetLensDistance();
	float newDistance = m_lensDistanceAnim.FinalVal + kLensMovementStep;

	// Validate and animate
	newDistance = m_lens->ValidateLensDistance(newDistance);
	AnimateCameraParam(m_lensDistanceAnim, initial, newDistance);
	
	// Print debug info
	float newFocalDistance = m_lens->CalcFocalDistance(m_lens->GetFocalLength(), newDistance);
	DebugPrint(L"Lens is now at %f mm from film, focused at a distance of %f m.\n", newDistance * kMetersToMm, newFocalDistance);
}

void Camera::MoveLensBack()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initial = m_lens->GetLensDistance();
	float newDistance = m_lensDistanceAnim.FinalVal - kLensMovementStep;
	
	// Validate and animate
	newDistance = m_lens->ValidateLensDistance(newDistance);
	AnimateCameraParam(m_lensDistanceAnim, initial, newDistance);

	// Print debug info
	float newFocalDistance = m_lens->CalcFocalDistance(m_lens->GetFocalLength(), newDistance);
	DebugPrint(L"Lens is now at %f mm from film, focused at a distance of %f m.\n", newDistance * kMetersToMm, newFocalDistance);
}

void Camera::MoveFocusForward()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initialLensDistance = m_lens->GetLensDistance();
	float curFocalDistance = m_lens->CalcFocalDistance(m_lens->GetFocalLength(), m_lensDistanceAnim.FinalVal);

	// Increment based on current distance
	float newFocalDistance = curFocalDistance;
	bool incremented = false;
	ASSERT(ARRAYSIZE(kFocalDistanceThresholds) == ARRAYSIZE(kFocalDistanceSteps));
	int numThresholds = ARRAYSIZE(kFocalDistanceThresholds);
	// Find the threshold range in which the current value falls, then adjust accordingly
	for(int i = 0; i < numThresholds; ++i) {
		if(curFocalDistance < kFocalDistanceThresholds[i]) {
			newFocalDistance += kFocalDistanceSteps[i];
			incremented = true;
			break;
		}
	}
	if(!incremented) {
		newFocalDistance += kLargestFocalDistanceStep;
	}

	// Validate and animate
	newFocalDistance = m_lens->ValidateFocalDistance(newFocalDistance);
	// Convert back to lens distance so it can be animated
	float newLensDistance = m_lens->CalcLensDistance(m_lens->GetFocalLength(), newFocalDistance);

	AnimateCameraParam(m_lensDistanceAnim, initialLensDistance, newLensDistance);
	DebugPrint(L"Lens is now at %f mm from film, focused at a distance of %f m.\n", newLensDistance * kMetersToMm, newFocalDistance);
}

void Camera::MoveFocusBack()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initialLensDistance = m_lens->GetLensDistance();
	float curFocalDistance = m_lens->CalcFocalDistance(m_lens->GetFocalLength(), m_lensDistanceAnim.FinalVal);

	// Decrement based on current distance
	float newFocalDistance = curFocalDistance;
	bool decremented = false;
	ASSERT(ARRAYSIZE(kFocalDistanceThresholds) == ARRAYSIZE(kFocalDistanceSteps));
	int numThresholds = ARRAYSIZE(kFocalDistanceThresholds);
	for(int i = 0; i < numThresholds; ++i) {
		if(curFocalDistance <= kFocalDistanceThresholds[i]) {
			newFocalDistance -= kFocalDistanceSteps[i];
			decremented = true;
			break;
		}
	}
	if(!decremented) {
		newFocalDistance -= kLargestFocalDistanceStep;
	}

	// Validate and animate
	newFocalDistance = m_lens->ValidateFocalDistance(newFocalDistance);
	// Convert back to lens distance so it can be animated
	float newLensDistance = m_lens->CalcLensDistance(m_lens->GetFocalLength(), newFocalDistance);

	AnimateCameraParam(m_lensDistanceAnim, initialLensDistance, newLensDistance);
	DebugPrint("Lens is now at %f mm from film, focused at a distance of %f m.\n", newLensDistance * kMetersToMm, newFocalDistance);
}

void Camera::DecreaseFNumber()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initial = m_aperture->GetFNumber();

	// Look through the array of F-stops and use the next smallest value
	float newFNumber = m_fNumberAnim.FinalVal;
	for(float fNumber : kFThirdStops) {
		if(fNumber >= m_fNumberAnim.FinalVal) {
			break;
		}
		
		// The f number at this index is still smaller than our current f number
		newFNumber = fNumber;
	}

	// Validate and animate
	newFNumber = m_aperture->ValidateFNumber(newFNumber);
	AnimateCameraParam(m_fNumberAnim, initial, newFNumber);
	DebugPrint("F-number is now %f.\n", newFNumber);
}

void Camera::IncreaseFNumber()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initial = m_aperture->GetFNumber();
	
	// Look through the array of F-stops and use the next largest value
	float newFNumber = m_fNumberAnim.FinalVal;
	for(float fNumber : kFThirdStops) {
		if(fNumber > newFNumber) {
			newFNumber = fNumber;
			break;
		}
	}

	// Validate and animate
	newFNumber = m_aperture->ValidateFNumber(newFNumber);
	AnimateCameraParam(m_fNumberAnim, initial, newFNumber);
	DebugPrint("F-number is now %f.\n", newFNumber);
}

void Camera::IncreaseLensFocalLength()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initial = m_lens->GetFocalLength();
	float newFocalLength = m_focalLengthAnim.FinalVal + kLensFocalLengthStep;

	// Validate and animate
	newFocalLength = m_lens->ValidateFocalLength(newFocalLength);
	AnimateCameraParam(m_focalLengthAnim, initial, newFocalLength);
	DebugPrint("Setting the lens focal length to %fmm.\n", newFocalLength * kMetersToMm);

	// Reset lens distance animations
	m_lensDistanceAnim.InitialVal = m_lens->GetLensDistance();
	m_lensDistanceAnim.FinalVal = m_lens->GetLensDistance();
	m_lensDistanceAnim.TimeLeft = 0;
	m_lensDistanceAnim.TotalTime = 0;
}

void Camera::DecreaseLensFocalLength()
{
	// Continue the animation based off the previous value
	// to handle a new value being set in the middle of an animation
	float initial = m_lens->GetFocalLength();
	float newFocalLength = m_focalLengthAnim.FinalVal - kLensFocalLengthStep;

	// Validate and animate
	newFocalLength = m_lens->ValidateFocalLength(newFocalLength);
	AnimateCameraParam(m_focalLengthAnim, initial, newFocalLength);
	DebugPrint("Setting the lens focal length to %fmm.\n", newFocalLength * kMetersToMm);

	// Reset lens distance animations
	m_lensDistanceAnim.InitialVal = m_lens->GetLensDistance();
	m_lensDistanceAnim.FinalVal = m_lens->GetLensDistance();
	m_lensDistanceAnim.TimeLeft = 0;
	m_lensDistanceAnim.TotalTime = 0;
}

void Camera::IncreaseShutterSpeed()
{
	float newShutterSpeed = m_shutter->GetShutterSpeed() * kShutterSpeedMultiplier;
	m_shutter->SetShutterSpeed(newShutterSpeed);

	DebugPrint(L"Shutter speed is now %f sec.\n", m_shutter->GetShutterSpeed());
}

void Camera::DecreaseShutterSpeed()
{
	float newShutterSpeed = m_shutter->GetShutterSpeed() / kShutterSpeedMultiplier;
	m_shutter->SetShutterSpeed(newShutterSpeed);
	
	DebugPrint(L"Shutter speed is now %f sec.\n", m_shutter->GetShutterSpeed());
}

void Camera::IncreaseFilmSpeed()
{
	int newFilmSpeed = m_film->GetFilmSpeed() * kFilmSpeedMultiplier;
	m_film->SetFilmSpeed(newFilmSpeed);
	
	DebugPrint(L"Film speed is now ISO %d.\n", m_film->GetFilmSpeed());
}

void Camera::DecreaseFilmSpeed()
{
	int newFilmSpeed = m_film->GetFilmSpeed() / kFilmSpeedMultiplier;
	m_film->SetFilmSpeed(newFilmSpeed);

	DebugPrint(L"Film speed is now ISO %d.\n", m_film->GetFilmSpeed());
}

void Camera::AnimateCameraParam(AnimParams &params, float initialValue, float newValue)
{
	// HACK: Make sure that the parameter is set even when the animation duration is 0 by applying an epsilon.
	float duration = std::max(kCameraAnimationDuration, 0.00001f);

	params.InitialVal = initialValue;
	params.FinalVal = newValue;
	params.TotalTime = kCameraAnimationDuration;
	params.TimeLeft = kCameraAnimationDuration;
}

void Camera::FixedUpdate(float curTime, float delta)
{
	if(m_fNumberAnim.TimeLeft > 0) {
		m_fNumberAnim.TimeLeft -= delta;
		float t = m_fNumberAnim.TimeLeft / m_fNumberAnim.TotalTime;
		float newFNumber = ClampedLerp(m_fNumberAnim.FinalVal, m_fNumberAnim.InitialVal, t);
		m_aperture->SetFNumber(newFNumber);
	}
	if(m_lensDistanceAnim.TimeLeft > 0) {
		m_lensDistanceAnim.TimeLeft -= delta;
		float t = m_lensDistanceAnim.TimeLeft / m_lensDistanceAnim.TotalTime;
		float newLensDistance = ClampedLerp(m_lensDistanceAnim.FinalVal, m_lensDistanceAnim.InitialVal, t);
		m_lens->SetLensDistance(newLensDistance);
	}
	if(m_focalLengthAnim.TimeLeft > 0) {
		m_focalLengthAnim.TimeLeft -= delta;
		float t = m_focalLengthAnim.TimeLeft / m_focalLengthAnim.TotalTime;
		float newFocalLength = ClampedLerp(m_focalLengthAnim.FinalVal, m_focalLengthAnim.InitialVal, t);
		m_lens->SetFocalLength(newFocalLength);
	}
}

Camera::AnimParams::AnimParams() :
	TimeLeft(0),
	TotalTime(0),
	InitialVal(0),
	FinalVal(0)
{
}

float Camera::CalculateGrainStrength()
{
	// Grain should be stronger at higher ISOs
	// Currently not tied to specific physical measurements (could maybe use SNR?)

	// Max film speed is currently 3200, tie it to highest film grain amount
	float modifier = static_cast<float>(m_film->GetFilmSpeed()) / 3200;
	const float maxGrainAmt = 0.1f;
	return modifier * maxGrainAmt;
}

float Camera::CalculateAngleOfView( float lengthAlongDimension )
{
	// Angle of view = 2 * arctan * (dimension / (2 * v))
	float v = m_lens->GetLensDistance();
	float sizeRatio = lengthAlongDimension / (2 * v);
	return 2 * atanf(sizeRatio);
}

void Camera::UpdateViewMatrix(_Out_ XMMATRIX& view)
{
	auto e = m_entity.lock();
	ASSERT(e != nullptr);
	auto t = e->GetTransform();

	// Load vectors from memory
	// TODO: Support camera roll (changing up vector)
	XMVECTOR cameraPos = XMLoadFloat3A(&t->GetLocalPosition());
	XMVECTOR forward = t->GetForwardVector();
	XMVECTOR up = XMVectorSet(0.0f, 1.0f, 0.0f, 1.0f);

	// Calculate the view matrix
	view = XMMatrixLookToLH(cameraPos, forward, up);
}

void Camera::UpdateLensParameters(XMINT2 renderSize, _In_ const XMMATRIX& view, _Out_ XMMATRIX& projection, _Out_ XMMATRIX& viewProjection)
{
	float aspectHByW = static_cast<float>(renderSize.x) / static_cast<float>(renderSize.y);

	// Default values, assume a 50mm lens at f/16 (focused close to infinity)
	float focalLength = m_lens->GetFocalLength();
	float focalDistance = m_lens->GetFocalDistance();
	float lensDistance = m_lens->GetLensDistance();
	float fNumber = m_aperture->GetFNumber();
	
	// Arbitrary near and far clips
	float nearClip = kNearClip;
	float farClip = kFarClip;

	// The lens is focused at infinity when it's at (focal length) distance from the film.
	// For now, make sure they aren't exactly equal because the DoF equation won't work
	// (the limit will need to be taken to prevent division by 0).
	ASSERT(focalLength != focalDistance);

	// Calculate the fov from camera parameters
	XMFLOAT2 filmSize = CalculateFilmSize(aspectHByW);
	float fovAngleY = CalculateVerticalAngleOfView(aspectHByW);

	// Calculate the projection and viewProjection matrices
	projection = XMMatrixPerspectiveFovLH(fovAngleY, aspectHByW, nearClip, farClip);
	viewProjection = XMMatrixMultiply(view, projection);

	//--------------------------------------------------------------------------------------
	// Update the constant buffer

	// Update the view-projection matrix
	XMStoreFloat4x4A(&m_cameraCBufferData.ViewProjection, XMMatrixTranspose(viewProjection));
	
	// Store clip distances
	m_cameraCBufferData.NearClip = nearClip;
	m_cameraCBufferData.FarClip = farClip;
	
	// Store projection constants (for getting view space position from depth values)
	float projectionA = farClip / (farClip - nearClip);
	float projectionB = (-farClip * nearClip) / (farClip - nearClip);
	m_cameraCBufferData.ProjectionA = projectionA;
	m_cameraCBufferData.ProjectionB = projectionB;

	// Store depth of field parameters
	m_cameraCBufferData.FocalLength = focalLength;
	m_cameraCBufferData.FocalDistance = focalDistance;
	m_cameraCBufferData.LensDistance = lensDistance;
	m_cameraCBufferData.FNumber = fNumber;
	m_cameraCBufferData.FilmWidth = filmSize.x;
	m_cameraCBufferData.FilmHeight = filmSize.y;
	m_cameraCBufferData.ImageWidth = renderSize.x;
	m_cameraCBufferData.ImageHeight = renderSize.y;

	// Store aberration parameters
	m_cameraCBufferData.SphericalAberrationFactor = m_lens->m_sphericalAberrationFactor;
}

void Camera::UpdateExposureParameters( float curTime )
{
	// Get exposure modifier
	float exposureModifier = CalculateExposureModifier();

	// Store exposure modifier
	m_cameraCBufferData.ExposureModifier = exposureModifier;

	// Store film grain modifiers
	m_cameraCBufferData.Time = curTime;
	m_cameraCBufferData.GrainSpeed = 0.1f;
	m_cameraCBufferData.NoiseStrength = CalculateGrainStrength();
}

void Camera::UpdateConstantBuffer( XMINT2 renderSize, float curTime, float delta, _Out_ XMMATRIX& view, _Out_ XMMATRIX& proj, _Out_ XMMATRIX& viewProj)
{
	UpdateViewMatrix(view);
	UpdateLensParameters(renderSize, view, proj, viewProj);
	UpdateExposureParameters(curTime);
}

} // end namespace